'use strict';

require('@shopify/shopify-api/adapters/web-api');
var shopifyApi = require('@shopify/shopify-api');
var types = require('./types.js');
var version = require('./version.js');
var register = require('./authenticate/webhooks/register.js');
var authenticate = require('./authenticate/admin/authenticate.js');
var authenticate$1 = require('./authenticate/webhooks/authenticate.js');
var overrideLogger = require('./override-logger.js');
var addResponseHeaders = require('./authenticate/helpers/add-response-headers.js');
require('@remix-run/server-runtime');
require('isbot');
var login = require('./authenticate/login/login.js');
var factory$2 = require('./unauthenticated/admin/factory.js');
var factory = require('./authenticate/public/factory.js');
var factory$1 = require('./unauthenticated/storefront/factory.js');
var authCodeFlow = require('./authenticate/admin/strategies/auth-code-flow.js');
var tokenExchange = require('./authenticate/admin/strategies/token-exchange.js');
var merchantCustomApp = require('./authenticate/admin/strategies/merchant-custom-app.js');
var idempotentPromiseHandler = require('./authenticate/helpers/idempotent-promise-handler.js');
var authenticate$3 = require('./authenticate/flow/authenticate.js');
var authenticate$2 = require('./authenticate/fulfillment-service/authenticate.js');
var flags = require('./future/flags.js');

/**
 * Creates an object your app will use to interact with Shopify.
 *
 * @param appConfig Configuration options for your Shopify app, such as the scopes your app needs.
 * @returns `ShopifyApp` An object constructed using your appConfig.  It has methods for interacting with Shopify.
 *
 * @example
 * <caption>The minimum viable configuration</caption>
 * ```ts
 * // /shopify.server.ts
 * import { shopifyApp } from "@shopify/shopify-app-remix/server";
 *
 * const shopify = shopifyApp({
 *   apiKey: process.env.SHOPIFY_API_KEY!,
 *   apiSecretKey: process.env.SHOPIFY_API_SECRET!,
 *   scopes: process.env.SCOPES?.split(",")!,
 *   appUrl: process.env.SHOPIFY_APP_URL!,
 * });
 * export default shopify;
 * ```
 */
function shopifyApp(appConfig) {
    const api = deriveApi(appConfig);
    const config = deriveConfig(appConfig, api.config);
    const logger = overrideLogger.overrideLogger(api.logger);
    if (appConfig.webhooks) {
        api.webhooks.addHandlers(appConfig.webhooks);
    }
    const params = { api, config, logger };
    let strategy;
    if (config.distribution === types.AppDistribution.ShopifyAdmin) {
        strategy = new merchantCustomApp.MerchantCustomAuth(params);
    }
    else if (config.future.unstable_newEmbeddedAuthStrategy &&
        config.isEmbeddedApp) {
        strategy = new tokenExchange.TokenExchangeStrategy(params);
    }
    else {
        strategy = new authCodeFlow.AuthCodeFlowStrategy(params);
    }
    const authStrategy = authenticate.authStrategyFactory({
        ...params,
        strategy,
    });
    const shopify = {
        sessionStorage: config.sessionStorage,
        addDocumentResponseHeaders: addResponseHeaders.addDocumentResponseHeadersFactory(params),
        registerWebhooks: register.registerWebhooksFactory(params),
        authenticate: {
            admin: authStrategy,
            flow: authenticate$3.authenticateFlowFactory(params),
            public: factory.authenticatePublicFactory(params),
            fulfillmentService: authenticate$2.authenticateFulfillmentServiceFactory(params),
            webhook: authenticate$1.authenticateWebhookFactory(params),
        },
        unauthenticated: {
            admin: factory$2.unauthenticatedAdminContextFactory(params),
            storefront: factory$1.unauthenticatedStorefrontContextFactory(params),
        },
    };
    if (isAppStoreApp(shopify, appConfig) ||
        isSingleMerchantApp(shopify, appConfig)) {
        shopify.login = login.loginFactory(params);
    }
    flags.logDisabledFutureFlags(config, logger);
    return shopify;
}
function isAppStoreApp(_shopify, config) {
    return config.distribution === types.AppDistribution.AppStore;
}
function isSingleMerchantApp(_shopify, config) {
    return config.distribution === types.AppDistribution.SingleMerchant;
}
// This function is only exported so we can unit test it without having to mock the underlying module.
// It's not available to consumers of the library because it is not exported in the index module, and never should be.
function deriveApi(appConfig) {
    let appUrl;
    try {
        appUrl = new URL(appConfig.appUrl);
    }
    catch (error) {
        const message = appConfig.appUrl === ''
            ? `Detected an empty appUrl configuration, please make sure to set the necessary environment variables.\n` +
                `If you're deploying your app, you can find more information at https://shopify.dev/docs/apps/launch/deployment/deploy-web-app/deploy-to-hosting-service#step-4-set-up-environment-variables`
            : `Invalid appUrl configuration '${appConfig.appUrl}', please provide a valid URL.`;
        throw new shopifyApi.ShopifyError(message);
    }
    /* eslint-disable no-process-env */
    if (appUrl.hostname === 'localhost' && !appUrl.port && process.env.PORT) {
        appUrl.port = process.env.PORT;
    }
    /* eslint-enable no-process-env */
    appConfig.appUrl = appUrl.origin;
    let userAgentPrefix = `Shopify Remix Library v${version.SHOPIFY_REMIX_LIBRARY_VERSION}`;
    if (appConfig.userAgentPrefix) {
        userAgentPrefix = `${appConfig.userAgentPrefix} | ${userAgentPrefix}`;
    }
    return shopifyApi.shopifyApi({
        ...appConfig,
        hostName: appUrl.host,
        hostScheme: appUrl.protocol.replace(':', ''),
        userAgentPrefix,
        isEmbeddedApp: appConfig.isEmbeddedApp ?? true,
        apiVersion: appConfig.apiVersion ?? shopifyApi.LATEST_API_VERSION,
        isCustomStoreApp: appConfig.distribution === types.AppDistribution.ShopifyAdmin,
        billing: appConfig.billing,
        future: {
            lineItemBilling: true,
            unstable_managedPricingSupport: true,
        },
        _logDisabledFutureFlags: false,
    });
}
function deriveConfig(appConfig, apiConfig) {
    if (!appConfig.sessionStorage &&
        appConfig.distribution !== types.AppDistribution.ShopifyAdmin) {
        throw new shopifyApi.ShopifyError('Please provide a valid session storage. Refer to https://github.com/Shopify/shopify-app-js/blob/main/README.md#session-storage-options for options.');
    }
    const authPathPrefix = appConfig.authPathPrefix || '/auth';
    appConfig.distribution = appConfig.distribution ?? types.AppDistribution.AppStore;
    return {
        ...appConfig,
        ...apiConfig,
        billing: appConfig.billing,
        scopes: apiConfig.scopes,
        idempotentPromiseHandler: new idempotentPromiseHandler.IdempotentPromiseHandler(),
        canUseLoginForm: appConfig.distribution !== types.AppDistribution.ShopifyAdmin,
        useOnlineTokens: appConfig.useOnlineTokens ?? false,
        hooks: appConfig.hooks ?? {},
        sessionStorage: appConfig.sessionStorage,
        future: appConfig.future ?? {},
        auth: {
            path: authPathPrefix,
            callbackPath: `${authPathPrefix}/callback`,
            patchSessionTokenPath: `${authPathPrefix}/session-token`,
            exitIframePath: `${authPathPrefix}/exit-iframe`,
            loginPath: `${authPathPrefix}/login`,
        },
        distribution: appConfig.distribution,
    };
}

exports.deriveApi = deriveApi;
exports.shopifyApp = shopifyApp;
//# sourceMappingURL=shopify-app.js.map
