import { BillingError, GraphqlQueryError } from '../error.mjs';
import { graphqlClientClass } from '../clients/admin/graphql/client.mjs';
import '@shopify/admin-api-client';
import '@shopify/network';
import '../types.mjs';
import '../../runtime/crypto/crypto.mjs';
import '../../runtime/crypto/types.mjs';
import 'compare-versions';
import { assessPayments } from './check.mjs';
import { convertAppRecurringPricingMoney, convertAppUsagePricingMoney } from './utils.mjs';

const CREATE_USAGE_RECORD_MUTATION = `
mutation appUsageRecordCreate($description: String!, $price: MoneyInput!, $subscriptionLineItemId: ID!) {
  appUsageRecordCreate(description: $description, price: $price, subscriptionLineItemId: $subscriptionLineItemId) {
    userErrors {
      field
      message
    }
    appUsageRecord {
      id
      description
      idempotencyKey
      price {
        amount
        currencyCode
      }
      subscriptionLineItem {
        id
        plan {
          pricingDetails {
            ... on AppUsagePricing {
              balanceUsed {
                amount
                currencyCode
              }
              cappedAmount {
                amount
                currencyCode
              }
              terms
            }
          }
        }
      }
    }
  }
}
`;
function createUsageRecord(config) {
    return async function createUsageRecord(usageRecordInfo) {
        const { session, subscriptionLineItemId, description, price, idempotencyKey, isTest = true, } = usageRecordInfo;
        const GraphqlClient = graphqlClientClass({ config });
        const client = new GraphqlClient({ session });
        // If a subscription line item ID is not passed, we will query Shopify
        // for an active usage subscription line item ID
        const usageSubscriptionLineItemId = subscriptionLineItemId
            ? subscriptionLineItemId
            : await getUsageRecordSubscriptionLineItemId({ client, isTest });
        const variables = {
            description,
            price,
            subscriptionLineItemId: usageSubscriptionLineItemId,
        };
        if (idempotencyKey) {
            variables.idempotencyKey = idempotencyKey;
        }
        try {
            const response = await client.request(CREATE_USAGE_RECORD_MUTATION, {
                variables,
            });
            if (response.data?.appUsageRecordCreate?.userErrors.length) {
                throw new BillingError({
                    message: 'Error while creating a usage record',
                    errorData: response.data?.appUsageRecordCreate?.userErrors,
                });
            }
            const appUsageRecord = response.data?.appUsageRecordCreate?.appUsageRecord;
            convertAppRecurringPricingMoney(appUsageRecord.price);
            convertAppUsagePricingMoney(appUsageRecord.subscriptionLineItem.plan.pricingDetails);
            return appUsageRecord;
        }
        catch (error) {
            if (error instanceof GraphqlQueryError) {
                throw new BillingError({
                    message: error.message,
                    errorData: error.response?.errors,
                });
            }
            else {
                throw error;
            }
        }
    };
}
async function getUsageRecordSubscriptionLineItemId({ client, isTest, }) {
    const payments = await assessPayments({ client, isTest });
    if (!payments.hasActivePayment) {
        throw new BillingError({
            message: 'No active payment found',
            errorData: [],
        });
    }
    if (!payments.appSubscriptions.length) {
        throw new BillingError({
            message: 'No active subscriptions found',
            errorData: [],
        });
    }
    if (payments.appSubscriptions) {
        const usageSubscriptionLineItemId = getUsageLineItemId(payments.appSubscriptions);
        return usageSubscriptionLineItemId;
    }
    throw new BillingError({
        message: 'Unable to find active subscription line item',
        errorData: [],
    });
}
function getUsageLineItemId(subscriptions) {
    for (const subscription of subscriptions) {
        // An app can have only one active subscription
        if (subscription.status === 'ACTIVE' && subscription.lineItems) {
            // An app can have only one usage subscription line item
            for (const lineItem of subscription.lineItems) {
                if ('balanceUsed' in lineItem.plan.pricingDetails) {
                    return lineItem.id;
                }
            }
        }
    }
    throw new BillingError({
        message: 'No active usage subscription found',
        errorData: [],
    });
}

export { createUsageRecord };
//# sourceMappingURL=create-usage-record.mjs.map
