'use strict';

var polarisTokens = require('@shopify/polaris-tokens');
var isObject = require('./is-object.js');

function classNames(...classes) {
  return classes.filter(Boolean).join(' ');
}
function variationName(name, value) {
  return `${name}${value.charAt(0).toUpperCase()}${value.slice(1)}`;
}
function sanitizeCustomProperties(styles) {
  const nonNullValues = Object.entries(styles).filter(([_, value]) => value != null);
  return nonNullValues.length ? Object.fromEntries(nonNullValues) : undefined;
}

/**
 * Given params like so:
 * (
 *   'button',
 *   'padding',
 *   'spacing',
 *   {
 *     sm: "4",
 *     lg: "6"
 *   }
 * )
 * Converts it to an object like so:
 * {
 *   '--pc-button-padding-sm': 'var(--p-spacing-4)',
 *   '--pc-button-padding-lg': 'var(--p-spacing-6)'
 * }
 *
 */
function getResponsiveProps(componentName, componentProp, tokenSubgroup, responsiveProp) {
  if (!responsiveProp) return {};
  let result;
  if (!isObject.isObject(responsiveProp)) {
    result = {
      [polarisTokens.breakpointsAliases[0]]: `var(--p-${tokenSubgroup}-${responsiveProp})`
    };
  } else {
    result = Object.fromEntries(Object.entries(responsiveProp).map(([breakpointAlias, aliasOrScale]) => [breakpointAlias, `var(--p-${tokenSubgroup}-${aliasOrScale})`]));
  }

  // Prefix each responsive key with the correct token name
  return Object.fromEntries(Object.entries(result).map(([breakpointAlias, value]) => [`--pc-${componentName}-${componentProp}-${breakpointAlias}`, value]));
}
function getResponsiveValue(componentName, componentProp, responsiveProp) {
  if (!responsiveProp) return {};
  if (!isObject.isObject(responsiveProp)) {
    return {
      [`--pc-${componentName}-${componentProp}-${polarisTokens.breakpointsAliases[0]}`]: responsiveProp
    };
  }
  return Object.fromEntries(Object.entries(responsiveProp).map(([breakpointAlias, responsiveValue]) => [`--pc-${componentName}-${componentProp}-${breakpointAlias}`, responsiveValue]));
}

exports.classNames = classNames;
exports.getResponsiveProps = getResponsiveProps;
exports.getResponsiveValue = getResponsiveValue;
exports.sanitizeCustomProperties = sanitizeCustomProperties;
exports.variationName = variationName;
