import { json } from "@remix-run/node";
import prisma from "../db.server";

export const loader = async ({ request }) => {
  try {
    const url = new URL(request.url);
    const userIdRaw = url.searchParams.get("userId");
    const product_id = url.searchParams.get("productId") || "";

    // validation: userId must be present and an integer
    const userId = parseInt(userIdRaw, 10);
    if (!userIdRaw || Number.isNaN(userId)) {
      return json({ error: "Invalid or missing userId" }, { status: 400 });
    }

    // parse product IDs into an array (strings). Filter empties.
    const productIdArray = product_id
      .split(",")
      .map(id => id.trim())
      .filter(id => id !== "");

    if (productIdArray.length === 0) {
      return json({ error: "No productId provided" }, { status: 400 });
    }

    // Fetch user (to get skulimit) — handle not found
    const user = await prisma.users.findUnique({
      where: { id: userId },
      select: { id: true, skuconsumed: true, skulimit: true },
    });

    if (!user) {
      return json({ error: "User not found" }, { status: 404 });
    }

    const countToConsume = productIdArray.length;

    // Attempt atomic conditional increment using updateMany (prevents race)
    // Ensure we only increment when skuconsumed + countToConsume <= skulimit
    // Prisma doesn't support arithmetic in 'where', so do: skuconsumed <= skulimit - countToConsume
    const maxAllowedConsumed = user.skulimit - countToConsume;

    if (maxAllowedConsumed < 0) {
      // request itself exceeds total limit
      return json(
        { msg: "Requested import count exceeds your total SKU limit." },
        { status: 403 }
      );
    }

    const updatedUsers = await prisma.users.updateMany({
      where: {
        id: userId,
        skuconsumed: { lte: maxAllowedConsumed }, // ensure there's room
      },
      data: {
        skuconsumed: { increment: countToConsume },
      },
    });

    if (updatedUsers.count === 0) {
      // no rows updated → user is at/over limit (likely due to concurrency)
      return json(
        {
          msg: "You have reached your import limit. Please upgrade your plan to import more products.",
        },
        { status: 403 }
      );
    }

    // Now mark products as import in progress
    const updateProducts = await prisma.products.updateMany({
      where: {
        product_id: { in: productIdArray },
      },
      data: {
        status: "Import_in_progress",
      },
    });

    // Optional: check how many were updated vs requested
    if (updateProducts.count === 0) {
      // nothing updated — maybe product IDs don't exist
      // You could choose to roll back the user increment in this case, depending on business rules.
      return json(
        {
          success: false,
          message: "No products updated. Please check product IDs.",
          productsRequested: productIdArray.length,
          productsUpdated: updateProducts.count,
        },
        { status: 404 }
      );
    }

    return json({
      success: true,
      message: "Product IDs received and import started",
      productsRequested: productIdArray.length,
      productsUpdated: updateProducts.count,
    });
  } catch (error) {
    console.error("Count API Error:", error);
    return json({ error: error.message }, { status: 500 });
  }
};
