import { createSHA256HMAC } from '../crypto/utils.mjs';
import { splitN } from './utils.mjs';
import { getHeader, getHeaders, removeHeader, addHeader } from './headers.mjs';

// import type {Headers} from "./headers";
class Cookies {
    response;
    static parseCookies(hdrs) {
        const entries = hdrs
            .filter((hdr) => hdr.trim().length > 0)
            .map((cookieDef) => {
            const [keyval, ...opts] = cookieDef.split(';');
            const [name, value] = splitN(keyval, '=', 2).map((value) => value.trim());
            return [
                name,
                {
                    name,
                    value,
                    ...Object.fromEntries(opts.map((opt) => splitN(opt, '=', 2).map((value) => value.trim()))),
                },
            ];
        });
        const jar = Object.fromEntries(entries);
        for (const cookie of Object.values(jar)) {
            if (typeof cookie.expires === 'string') {
                cookie.expires = new Date(cookie.expires);
            }
        }
        return jar;
    }
    static encodeCookie(data) {
        let result = '';
        result += `${data.name}=${data.value};`;
        result += Object.entries(data)
            .filter(([key]) => !['name', 'value', 'expires'].includes(key))
            .map(([key, value]) => `${key}=${value}`)
            .join('; ');
        if (data.expires) {
            result += ';';
            result += `expires=${data.expires.toUTCString()}`;
        }
        return result;
    }
    receivedCookieJar = {};
    outgoingCookieJar = {};
    keys = [];
    constructor(request, response, { keys = [] } = {}) {
        this.response = response;
        if (keys)
            this.keys = keys;
        const cookieReqHdr = getHeader(request.headers, 'Cookie') ?? '';
        this.receivedCookieJar = Cookies.parseCookies(cookieReqHdr.split(';'));
        const cookieResHdr = getHeaders(response.headers, 'Set-Cookie') ?? [];
        this.outgoingCookieJar = Cookies.parseCookies(cookieResHdr);
    }
    toHeaders() {
        return Object.values(this.outgoingCookieJar).map((cookie) => Cookies.encodeCookie(cookie));
    }
    updateHeader() {
        if (!this.response.headers) {
            this.response.headers = {};
        }
        removeHeader(this.response.headers, 'Set-Cookie');
        this.toHeaders().map((hdr) => addHeader(this.response.headers, 'Set-Cookie', hdr));
    }
    get(name) {
        return this.receivedCookieJar[name]?.value;
    }
    deleteCookie(name) {
        this.set(name, '', {
            path: '/',
            expires: new Date(0),
        });
    }
    async getAndVerify(name) {
        const value = this.get(name);
        if (!value)
            return undefined;
        if (!(await this.isSignedCookieValid(name))) {
            return undefined;
        }
        return value;
    }
    get canSign() {
        return this.keys?.length > 0;
    }
    set(name, value, opts = {}) {
        this.outgoingCookieJar[name] = {
            ...opts,
            name,
            value,
        };
        this.updateHeader();
    }
    async setAndSign(name, value, opts = {}) {
        if (!this.canSign) {
            throw Error('No keys provided for signing.');
        }
        this.set(name, value, opts);
        const sigName = `${name}.sig`;
        const signature = await createSHA256HMAC(this.keys[0], value);
        this.set(sigName, signature, opts);
        this.updateHeader();
    }
    async isSignedCookieValid(cookieName) {
        const signedCookieName = `${cookieName}.sig`;
        if (!this.cookieExists(cookieName) ||
            !this.cookieExists(signedCookieName)) {
            this.deleteInvalidCookies(cookieName, signedCookieName);
            return false;
        }
        const cookieValue = this.get(cookieName);
        const signature = this.get(signedCookieName);
        if (!cookieValue || !signature) {
            this.deleteInvalidCookies(cookieName, signedCookieName);
            return false;
        }
        const allCheckSignatures = await Promise.all(this.keys.map((key) => createSHA256HMAC(key, cookieValue)));
        if (!allCheckSignatures.includes(signature)) {
            this.deleteInvalidCookies(cookieName, signedCookieName);
            return false;
        }
        return true;
    }
    cookieExists(cookieName) {
        return Boolean(this.get(cookieName));
    }
    deleteInvalidCookies(...cookieNames) {
        cookieNames.forEach((cookieName) => this.deleteCookie(cookieName));
    }
}

export { Cookies };
//# sourceMappingURL=cookies.mjs.map
