import { SESSION_COOKIE_NAME } from '../auth/oauth/types.mjs';
import { abstractConvertRequest } from '../../runtime/http/index.mjs';
import { sanitizeShop } from '../utils/shop-validator.mjs';
import { logger } from '../logger/index.mjs';
import { MissingJwtTokenError } from '../error.mjs';
import { decodeSessionToken } from './decode-session-token.mjs';
import { Session } from './session.mjs';
import { Cookies } from '../../runtime/http/cookies.mjs';

function getJwtSessionId(config) {
    return (shop, userId) => {
        return `${sanitizeShop(config)(shop, true)}_${userId}`;
    };
}
function getOfflineId(config) {
    return (shop) => {
        return `offline_${sanitizeShop(config)(shop, true)}`;
    };
}
function getCurrentSessionId(config) {
    return async function getCurrentSessionId({ isOnline, ...adapterArgs }) {
        const request = await abstractConvertRequest(adapterArgs);
        const log = logger(config);
        if (config.isEmbeddedApp) {
            log.debug('App is embedded, looking for session id in JWT payload', {
                isOnline,
            });
            const authHeader = request.headers.Authorization;
            if (authHeader) {
                const matches = (typeof authHeader === 'string' ? authHeader : authHeader[0]).match(/^Bearer (.+)$/);
                if (!matches) {
                    log.error('Missing Bearer token in authorization header', { isOnline });
                    throw new MissingJwtTokenError('Missing Bearer token in authorization header');
                }
                const jwtPayload = await decodeSessionToken(config)(matches[1]);
                const shop = jwtPayload.dest.replace(/^https:\/\//, '');
                log.debug('Found valid JWT payload', { shop, isOnline });
                if (isOnline) {
                    return getJwtSessionId(config)(shop, jwtPayload.sub);
                }
                else {
                    return getOfflineId(config)(shop);
                }
            }
            else {
                log.error('Missing Authorization header, was the request made with authenticatedFetch?', { isOnline });
            }
        }
        else {
            log.debug('App is not embedded, looking for session id in cookies', {
                isOnline,
            });
            const cookies = new Cookies(request, {}, {
                keys: [config.apiSecretKey],
            });
            return cookies.getAndVerify(SESSION_COOKIE_NAME);
        }
        return undefined;
    };
}
function customAppSession(config) {
    return (shop) => {
        return new Session({
            id: '',
            shop: `${sanitizeShop(config)(shop, true)}`,
            state: '',
            isOnline: false,
        });
    };
}

export { customAppSession, getCurrentSessionId, getJwtSessionId, getOfflineId };
//# sourceMappingURL=session-utils.mjs.map
