import * as addPlugin from '@graphql-codegen/add';
import * as typescriptPlugin from '@graphql-codegen/typescript';
import * as typescriptOperationPlugin from '@graphql-codegen/typescript-operations';
import { processSources } from './sources.js';
import { GENERATED_QUERY_INTERFACE_NAME, GENERATED_MUTATION_INTERFACE_NAME, plugin } from './plugin.js';

const ERROR_PREFIX = "[@shopify/graphql-codegen]";
const preset = {
  [Symbol.for("name")]: "@shopify/graphql-codegen",
  buildGeneratesSection: (options) => {
    if (!options.baseOutputDir.endsWith(".ts")) {
      throw new Error(
        ERROR_PREFIX + " target output should be a .ts or a .d.ts file."
      );
    }
    if (options.plugins?.length > 0 && Object.keys(options.plugins).some((p) => p.startsWith("typescript"))) {
      throw new Error(
        ERROR_PREFIX + " providing additional typescript-based `plugins` leads to duplicated generated types"
      );
    }
    const isDts = options.baseOutputDir.endsWith(".d.ts");
    const sourcesWithOperations = processSources(options.documents);
    const sources = sourcesWithOperations.map(({ source }) => source);
    const namespacedImportName = options.presetConfig.importTypes?.namespace;
    const importTypesFrom = options.presetConfig.importTypes?.from;
    const importTypes = !!namespacedImportName && !!importTypesFrom;
    const interfaceExtensionCode = options.presetConfig.interfaceExtension({
      queryType: GENERATED_QUERY_INTERFACE_NAME,
      mutationType: GENERATED_MUTATION_INTERFACE_NAME
    });
    const pluginMap = {
      ...options.pluginMap,
      [`add`]: addPlugin,
      [`typescript`]: typescriptPlugin,
      [`typescript-operations`]: typescriptOperationPlugin,
      [`gen-dts`]: { plugin: plugin }
    };
    const plugins = [
      // 1. Disable eslint for the generated file
      {
        [`add`]: {
          content: `/* eslint-disable eslint-comments/disable-enable-pair */
/* eslint-disable eslint-comments/no-unlimited-disable */
/* eslint-disable */`
        }
      },
      // 2. Import all the generated API types or generate all the types from the schema.
      importTypes ? {
        [`add`]: {
          content: `import${isDts ? " type " : " "}* as ${namespacedImportName} from '${importTypesFrom}';
`
        }
      } : {
        [`typescript`]: {
          enumsAsTypes: isDts,
          useTypeImports: isDts,
          useImplementingTypes: true
        }
      },
      // 3. Generate the operations (i.e. queries, mutations, and fragments types)
      {
        [`typescript-operations`]: {
          useTypeImports: isDts,
          // Use `import type` instead of `import`
          preResolveTypes: false,
          // Use Pick<...> instead of primitives
          mergeFragmentTypes: true,
          // Merge equal fragment interfaces. Avoids adding `| {}` to Metaobject
          skipTypename: options.presetConfig.skipTypenameInOperations ?? true,
          // Skip __typename fields
          namespacedImportName: importTypes ? namespacedImportName : void 0
        }
      },
      // 4.  Augment query/mutation interfaces with the generated operations
      { [`gen-dts`]: { sourcesWithOperations, interfaceExtensionCode } },
      // 5. Custom plugins from the user
      ...options.plugins
    ];
    return [
      {
        filename: options.baseOutputDir,
        plugins,
        pluginMap,
        schema: options.schema,
        config: {
          // For the TS plugin:
          defaultScalarType: "unknown",
          // Allow overwriting defaults:
          ...options.config
        },
        documents: sources,
        documentTransforms: options.documentTransforms
      }
    ];
  }
};

export { preset };
//# sourceMappingURL=out.js.map
//# sourceMappingURL=preset.js.map