import { forwardRef, useState, Children, useEffect } from 'react';
import ReactDOM from 'react-dom';
import { jsxs, jsx } from 'react/jsx-runtime';

/**
 * This component is a wrapper around the App Bridge `ui-modal` element.
 * It is used to display an overlay that prevents interaction with the
 * rest of the app until dismissed.
 *
 * @see {@link https://shopify.dev/docs/api/app-bridge-library/react-components/modal}
 */
const Modal = /*#__PURE__*/forwardRef(function InternalModal({
  open,
  onShow,
  onHide,
  children,
  ...rest
}, forwardedRef) {
  const [modal, setModal] = useState();
  const {
    titleBar,
    saveBar,
    modalContent
  } = Children.toArray(children).reduce((acc, node) => {
    const nodeName = getNodeName(node);
    const isTitleBar = nodeName === 'ui-title-bar';
    const isSaveBar = nodeName === 'ui-save-bar';
    const belongToModalContent = !isTitleBar && !isSaveBar;
    if (belongToModalContent) {
      acc.modalContent.push(node);
    }
    return {
      ...acc,
      titleBar: isTitleBar ? node : acc.titleBar,
      saveBar: isSaveBar ? node : acc.saveBar
    };
  }, {
    modalContent: []
  });
  const contentPortal = modal && modal.content ? /*#__PURE__*/ReactDOM.createPortal(modalContent, modal.content) : null;
  useEffect(() => {
    if (!modal) return;
    if (open) {
      modal.show();
    } else {
      modal.hide();
    }
  }, [modal, open]);
  useEffect(() => {
    if (!modal || !onShow) return;
    modal.addEventListener('show', onShow);
    return () => {
      modal.removeEventListener('show', onShow);
    };
  }, [modal, onShow]);
  useEffect(() => {
    if (!modal || !onHide) return;
    modal.addEventListener('hide', onHide);
    return () => {
      modal.removeEventListener('hide', onHide);
    };
  }, [modal, onHide]);
  useEffect(() => {
    if (!modal) return;
    return () => {
      modal.hide();
    };
  }, [modal]);
  return /*#__PURE__*/jsxs("ui-modal", {
    ...rest,
    ref: modal => {
      setModal(modal);
      if (forwardedRef) {
        if (typeof forwardedRef === 'function') {
          forwardedRef(modal);
        } else {
          forwardedRef.current = modal;
        }
      }
    },
    children: [titleBar, saveBar, /*#__PURE__*/jsx("div", {
      children: contentPortal
    })]
  });
});
Modal.displayName = 'ui-modal';
function getNodeName(node) {
  if (!node) return;
  const rawNodeType = typeof node === 'object' && 'type' in node ? node.type : undefined;
  const nodeType = typeof rawNodeType === 'string' ? rawNodeType : undefined;
  const rawDisplayName = typeof rawNodeType === 'object' ? rawNodeType.displayName : undefined;
  const displayName = typeof rawDisplayName === 'string' ? rawDisplayName : undefined;
  return nodeType || displayName;
}

export { Modal };
