import { json } from "@remix-run/node";
import prisma from "../db.server";

export const loader = async ({request}) => {
  try {
    const url = new URL(request.url);
    const userId = parseInt(url.searchParams.get("userId"));
    const page = parseInt(url.searchParams.get("page")) || 1;
    const pageSize = parseInt(url.searchParams.get("pageSize")) || 5;
    const status = url.searchParams.get("status") || null;

    const whereClause = {
      user_id: userId,
      ...(status ? { status } : {})
    };

    const [all, readyToImport, imported, importInProgress] = await Promise.all([
      prisma.products.count(),
      prisma.products.count({ where: { status: "Ready_to_Import" } }),
      prisma.products.count({ where: { status: "Imported" } }),
      prisma.products.count({ where: { status: "Import_in_progress" } }),
    ]);

    const total = await prisma.products.count({ where: whereClause });
    const skip = (page - 1) * pageSize;

    const products = await prisma.products.findMany({
      where: whereClause,
      skip,
      take: pageSize,
      orderBy: { created_at: "desc" },
      select: {
        product_id: true,
        title: true,
        status: true,
        description: true,
        created_at: true,
      },
    });

    return json({
      products,
      counts: {
        all,
        readyToImport,
        imported,
        importInProgress,
      },
      pagination: {
        page,
        pageSize,
        total,
        totalPages: Math.ceil(total / pageSize),
      },
    });
  } catch (error) {
    console.error("Count API Error:", error);
    return json({ error: error.message }, { status: 500 });
  }
};
